<?php
declare(strict_types=1);

// Prevent PHP from sending HTML errors to the client (API must return JSON only)
ini_set('display_errors', '0');
ini_set('log_errors', '1');

require __DIR__ . '/vendor/autoload.php';
if (file_exists(__DIR__ . '/.env')) {
    $dotenv = Dotenv\Dotenv::createImmutable(__DIR__);
    $dotenv->load();
}

$allowedOrigins = [
    'https://boazinfra.com',
    'https://www.boazinfra.com',
    'https://admin.boazinfra.com',
    'http://boazinfra.com',
    'http://www.boazinfra.com',
    'http://admin.boazinfra.com',
];
if (!empty($_ENV['CORS_ORIGINS'])) {
    $extra = array_map('trim', explode(',', $_ENV['CORS_ORIGINS']));
    $allowedOrigins = array_values(array_unique(array_merge($allowedOrigins, $extra)));
}
if (!empty($_ENV['CORS_ORIGIN'])) {
    $allowedOrigins[] = trim($_ENV['CORS_ORIGIN']);
    $allowedOrigins = array_values(array_unique($allowedOrigins));
}

$requestOrigin = $_SERVER['HTTP_ORIGIN'] ?? '';
$corsOrigin = in_array($requestOrigin, $allowedOrigins, true) ? $requestOrigin : ($allowedOrigins[0] ?? '*');

register_shutdown_function(function () use ($corsOrigin) {
    $err = error_get_last();
    if ($err && in_array($err['type'], [E_ERROR, E_PARSE, E_CORE_ERROR, E_COMPILE_ERROR], true)) {
        if (!headers_sent()) {
            header('Access-Control-Allow-Origin: ' . $corsOrigin);
            header('Content-Type: application/json; charset=UTF-8');
            http_response_code(500);
        }
        echo json_encode([
            'success' => false,
            'message' => 'Server error',
            'error' => $err['message'] ?? 'Unknown error',
            'file' => $err['file'] ?? null,
            'line' => $err['line'] ?? null,
        ]);
    }
});

header("Access-Control-Allow-Origin: $corsOrigin");
header("Access-Control-Allow-Methods: GET, POST, PUT, DELETE, OPTIONS");
header("Access-Control-Allow-Headers: Content-Type, Authorization");
header("Content-Type: application/json; charset=UTF-8");
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(204);
    exit;
}

function db(): PDO {
    static $pdo = null;
    if ($pdo === null) {
        $dsn = sprintf(
            'mysql:host=%s;dbname=%s;charset=utf8mb4',
            $_ENV['DB_HOST'] ?? '127.0.0.1',
            $_ENV['DB_NAME'] ?? 'bonza'
        );
        $pdo = new PDO($dsn, $_ENV['DB_USER'] ?? 'root', $_ENV['DB_PASS'] ?? '', [
            PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
            PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
        ]);
    }
    return $pdo;
}

/** Convert DB row to API-style user array (camelCase, _id/id as string) */
function rowToUser(array $row): array {
    if (empty($row)) return $row;
    $id = (string) ($row['id'] ?? '');
    $user = [
        '_id' => $id,
        'id' => $id,
        'number' => $row['number'] ?? '',
        'name' => $row['name'] ?? '',
        'email' => $row['email'] ?? '',
        'password' => $row['password'] ?? '',
        'dateOfBirth' => $row['date_of_birth'] ?? '',
        'referralId' => $row['referral_id'] ?? '',
        'referredBy' => $row['referred_by'] !== null ? (string) $row['referred_by'] : null,
        'status' => $row['status'] ?? 'pending',
        'role' => $row['role'] ?? 'user',
        'walletBalance' => (float) ($row['wallet_balance'] ?? 0),
        'minWithdrawBalance' => (float) ($row['min_withdraw_balance'] ?? 0),
        'profileImage' => $row['profile_image'] ?? null,
        'kycStatus' => $row['kyc_status'] ?? 'pending',
        'kycAadharNumber' => $row['kyc_aadhar_number'] ?? null,
        'kycAadharImage' => $row['kyc_aadhar_image'] ?? null,
        'kycPanNumber' => $row['kyc_pan_number'] ?? null,
        'kycPanImage' => $row['kyc_pan_image'] ?? null,
        'kycBankAccountNumber' => $row['kyc_bank_account_number'] ?? null,
        'kycBankIfsc' => $row['kyc_bank_ifsc'] ?? null,
        'kycBankName' => $row['kyc_bank_name'] ?? null,
        'kycAccountHolderName' => $row['kyc_account_holder_name'] ?? null,
        'kycSubmittedAt' => $row['kyc_submitted_at'] ?? null,
        'registrationTransactionId' => $row['registration_transaction_id'] ?? null,
        'registrationPaymentScreenshot' => $row['registration_payment_screenshot'] ?? null,
        'registrationSubmittedAt' => $row['registration_submitted_at'] ?? null,
        'approvedAt' => $row['approved_at'] ?? null,
        'createdAt' => $row['created_at'] ?? null,
    ];
    return $user;
}

function getJsonInput(): array {
    $input = file_get_contents('php://input');
    return json_decode($input ?: '{}', true) ?? [];
}

function getAuthUser(): ?array {
    $headers = getallheaders();
    $auth = $headers['Authorization'] ?? $headers['authorization'] ?? '';
    if (!preg_match('/Bearer\s+(.+)/', $auth, $m)) return null;
    try {
        $decoded = \Firebase\JWT\JWT::decode($m[1], new \Firebase\JWT\Key($_ENV['JWT_SECRET'] ?? 'secret', 'HS256'));
        $uid = $decoded->uid ?? null;
        if ($uid === null) return null;
        $stmt = db()->prepare('SELECT * FROM users WHERE id = ?');
        $stmt->execute([$uid]);
        $row = $stmt->fetch();
        return $row ? rowToUser($row) : null;
    } catch (\Throwable $e) {
        return null;
    }
}

function requireAuth(): array {
    $user = getAuthUser();
    if (!$user) {
        http_response_code(401);
        echo json_encode(['success' => false, 'message' => 'Unauthorized']);
        exit;
    }
    return $user;
}

function requireAdmin(): array {
    $user = requireAuth();
    if (empty($user['role']) || $user['role'] !== 'admin') {
        http_response_code(403);
        echo json_encode(['success' => false, 'message' => 'Forbidden']);
        exit;
    }
    return $user;
}

function jsonResponse(array $data): void {
    echo json_encode($data);
}

function generateReferralId(): string {
    $letters = 'ABCDEFGHIJKLMNOPQRSTUVWXYZ';
    $pdo = db();
    do {
        $id = $letters[random_int(0, 25)] . $letters[random_int(0, 25)] . sprintf('%04d', random_int(0, 9999));
        $stmt = $pdo->prepare('SELECT 1 FROM users WHERE referral_id = ?');
        $stmt->execute([$id]);
    } while ($stmt->fetch());
    return $id;
}

// Commission per level: 50%, 20%, 15%, 8%, 4%, 3% of 2000 = 1000, 400, 300, 160, 80, 60
const COMMISSION_PERCENT = [50, 20, 15, 8, 4, 3];
const COMMISSION_AMOUNTS = [1000, 400, 300, 160, 80, 60]; // from 2000 per 5000
const LEVEL_SIZE = 5;
const LEVELS = 6;
